(function() {
  const FLOW_COLORS = [
    "#4C6FFF", "#FF6B6B", "#F59F00", "#12B886", "#15AABF",
    "#845EF7", "#E64980", "#5C940D", "#FCC419", "#364FC7"
  ];

  const state = {
    schema: null,
    selectedType: null, // "schema" | "node" | "flow" | "text"
    selectedId: null,
    pan: { x: 0, y: 0 },
    zoom: 1,
    draggingPan: false,
    panMoved: false,
    tool: "hand",
    nodeSelection: new Set(),
    fileMode: null,
    nextFlowColorIndex: 0,
    // Arrow selection
    selecting: false,
    selectionStart: null,
    selectionRect: null,
    selectionAdditive: false,
    selectionJustFinished: false,
    // Node drag
    draggingNodes: false,
    dragNodeOffsets: null
  };

  let history = [];
  let contentGroup = null;

  const canvas = document.getElementById("canvas");
  const viewport = document.getElementById("viewport");
  const propertiesPanel = document.getElementById("propertiesPanel");
  const schemaTitleEl = document.getElementById("schemaTitle");
  const zoomLabel = document.getElementById("zoomLabel");

  const btnUndo = document.getElementById("btnUndo");
  const toolHandBtn = document.getElementById("toolHand");
  const toolArrowBtn = document.getElementById("toolArrow");
  const btnAddText = document.getElementById("btnAddText");
  const btnZoomIn = document.getElementById("btnZoomIn");
  const btnZoomOut = document.getElementById("btnZoomOut");
  const btnResetView = document.getElementById("btnResetView");
  const schemaMenu = document.getElementById("schemaMenu");
  const schemaMenuButton = document.getElementById("schemaMenuButton");
  const btnNew = document.getElementById("btnNew");
  const btnOpen = document.getElementById("btnOpen");
  const btnImport = document.getElementById("btnImport");
  const btnSaveFile = document.getElementById("btnSaveFile");
  const btnExportPng = document.getElementById("btnExportPng");
  const fileInput = document.getElementById("fileInput");
  const btnHelp = document.getElementById("btnHelp");
  const helpModal = document.getElementById("helpModal");
  const helpClose = document.getElementById("helpClose");

  /* ---------------- Model creation ---------------- */

  function createEmptySchema() {
    const now = new Date().toISOString();
    return {
      schema: {
        id: "schema-" + Date.now(),
        title: "Demo Value System",
        description: "",
        version: "1.0.0",
        author: "user",
        created_at: now,
        updated_at: now
      },
      nodes: [
        {
          id: "node-1",
          name: "Main Converter",
          position: { x: 0, y: 0 },
          size: { width: 120, height: 60 },
          description: "",
          link: "",
          attributes: {}
        }
      ],
      flows: [
        {
          id: "flow-1",
          name: "External Input",
          type: "external_in",
          source_node_id: null,
          target_node_id: "node-1",
          value: 100,
          unit: "Æ",
          color: FLOW_COLORS[0],
          show_label: false,
          attributes: {}
        },
        {
          id: "flow-2",
          name: "External Output",
          type: "external_out",
          source_node_id: "node-1",
          target_node_id: null,
          value: 100,
          unit: "Æ",
          color: FLOW_COLORS[1],
          show_label: false,
          attributes: {}
        }
      ],
      elements: [
        {
          id: "elem-1",
          name: "Customer Value",
          kind: "value",   // value | anti_value
          sign: 1,
          quantity: 100,
          unit: "Æ",
          location: "in_node",
          node_id: "node-1",
          flow_id: null,
          attributes: {}
        }
      ],
      texts: []
    };
  }

  function normalizeSchema(schema) {
    schema.flows = schema.flows || [];
    schema.elements = schema.elements || [];
    schema.texts = schema.texts || [];
    schema.flows.forEach(f => {
      if (!f.color) f.color = getNextFlowColor();
      if (typeof f.show_label !== "boolean") f.show_label = false;
      autoTypeFlow(f);
    });
    return schema;
  }

  function updateTimestamp() {
    state.schema.schema.updated_at = new Date().toISOString();
  }

  /* ---------------- Undo ---------------- */

  function pushHistory() {
    history.push(JSON.stringify(state.schema));
    if (history.length > 100) history.shift();
  }

  function undo() {
    if (!history.length) return;
    const snap = history.pop();
    state.schema = normalizeSchema(JSON.parse(snap));
    selectSchema();
    render();
  }

  btnUndo.onclick = undo;
  window.addEventListener("keydown", (e) => {
    if ((e.ctrlKey || e.metaKey) && e.key.toLowerCase() === "z") {
      e.preventDefault();
      undo();
    }
  });

  /* ---------------- Utils ---------------- */

  function getNodeById(id) {
    return state.schema.nodes.find(n => n.id === id) || null;
  }

  function getFlowById(id) {
    return state.schema.flows.find(f => f.id === id) || null;
  }

  function getTextById(id) {
    return state.schema.texts.find(t => t.id === id) || null;
  }

function getElementsInNode(nodeId) {
  return state.schema.elements.filter(
    e => e.location === "in_node" && e.node_id === nodeId
  );
}

  function generateId(prefix) {
    return prefix + "-" + Math.random().toString(36).slice(2, 8);
  }

  function autoTypeFlow(flow) {
    const hasSource = !!flow.source_node_id;
    const hasTarget = !!flow.target_node_id;
    if (hasSource && hasTarget) flow.type = "internal";
    else if (!hasSource && hasTarget) flow.type = "external_in";
    else if (hasSource && !hasTarget) flow.type = "external_out";
    else flow.type = "external_out";
  }

  function getNextFlowColor() {
    const color = FLOW_COLORS[state.nextFlowColorIndex % FLOW_COLORS.length];
    state.nextFlowColorIndex++;
    return color;
  }

  function snap(v, step = 10) {
    return Math.round(v / step) * step;
  }

  function clientToSvg(clientX, clientY) {
    const pt = canvas.createSVGPoint();
    pt.x = clientX;
    pt.y = clientY;
    const svgP = pt.matrixTransform(canvas.getScreenCTM().inverse());
    return {
      x: (svgP.x - state.pan.x) / state.zoom,
      y: (svgP.y - state.pan.y) / state.zoom
    };
  }

  function darkenColor(hex, amount) {
    if (!hex || hex[0] !== "#") return hex || "#999999";
    let r = parseInt(hex.slice(1, 3), 16);
    let g = parseInt(hex.slice(3, 5), 16);
    let b = parseInt(hex.slice(5, 7), 16);
    r = Math.max(0, Math.min(255, Math.round(r * (1 - amount))));
    g = Math.max(0, Math.min(255, Math.round(g * (1 - amount))));
    b = Math.max(0, Math.min(255, Math.round(b * (1 - amount))));
    return "#" + [r, g, b].map(v => v.toString(16).padStart(2, "0")).join("");
  }

  /* ---------------- Transform / zoom ---------------- */

  function applyTransform() {
    viewport.setAttribute(
      "transform",
      `translate(${state.pan.x}, ${state.pan.y}) scale(${state.zoom})`
    );
    zoomLabel.textContent = Math.round(state.zoom * 100) + "%";
  }

  btnZoomIn.onclick = () => {
    state.zoom = Math.min(3, state.zoom * 1.2);
    applyTransform();
  };

  btnZoomOut.onclick = () => {
    state.zoom = Math.max(0.3, state.zoom / 1.2);
    applyTransform();
  };

  btnResetView.onclick = () => {
    centerView();
  };

  canvas.addEventListener("wheel", (e) => {
    if (!e.ctrlKey) return;
    e.preventDefault();
    const delta = e.deltaY < 0 ? 1.1 : 0.9;
    state.zoom = Math.min(3, Math.max(0.3, state.zoom * delta));
    applyTransform();
  }, { passive: false });

  function centerView() {
    state.zoom = 1;
    state.pan = { x: canvas.clientWidth / 2, y: canvas.clientHeight / 2 };
    applyTransform();
  }

  /* ---------------- Render ---------------- */

  function refreshTitle() {
    schemaTitleEl.textContent = state.schema.schema.title || "(Untitled)";
  }

  function render() {
    renderDiagram();
    renderPropertiesPanel();
    refreshTitle();
  }

  function renderDiagram() {
    viewport.innerHTML = "";
    drawGrid();

    contentGroup = document.createElementNS("http://www.w3.org/2000/svg", "g");
    contentGroup.setAttribute("id", "content");
    viewport.appendChild(contentGroup);

    state.schema.flows.forEach(drawFlow);
    state.schema.nodes.forEach(drawNode);
    state.schema.texts.forEach(drawTextObject);
  }

  function drawGrid() {
    const group = document.createElementNS("http://www.w3.org/2000/svg", "g");
    group.setAttribute("id", "grid");
    group.setAttribute("opacity", "0.15");

    for (let x = -2000; x <= 2000; x += 40) {
      for (let y = -2000; y <= 2000; y += 40) {
        const c = document.createElementNS("http://www.w3.org/2000/svg", "circle");
        c.setAttribute("cx", x);
        c.setAttribute("cy", y);
        c.setAttribute("r", 0.6);
        c.setAttribute("fill", "#bcdcff");
        group.appendChild(c);
      }
    }
    viewport.appendChild(group);
  }

  /* ---------------- Text objects ---------------- */

  function drawTextObject(o) {
    const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
    g.setAttribute("class", "text-object");
    g.setAttribute("data-id", o.id);
    g.setAttribute("data-type", "text");

    const t = document.createElementNS("http://www.w3.org/2000/svg", "text");
    t.setAttribute("x", o.x);
    t.setAttribute("y", o.y);
    t.setAttribute("font-size", "13");
    t.setAttribute("fill", "#1f2933");
    t.textContent = o.text || "Comment";
    g.appendChild(t);

    let dragging = false;
    let offset = { x: 0, y: 0 };

    g.addEventListener("mousedown", (e) => {
      e.stopPropagation();
      const p = clientToSvg(e.clientX, e.clientY);
      offset.x = p.x - o.x;
      offset.y = p.y - o.y;
      dragging = true;
      state.selectedType = "text";
      state.selectedId = o.id;
      renderPropertiesPanel();
    });

    canvas.addEventListener("mousemove", (e) => {
      if (!dragging) return;
      const p = clientToSvg(e.clientX, e.clientY);
      o.x = snap(p.x - offset.x);
      o.y = snap(p.y - offset.y);
      renderDiagram();
    });

    window.addEventListener("mouseup", () => {
      if (dragging) {
        dragging = false;
        pushHistory();
      }
    });

    g.addEventListener("click", (e) => {
      e.stopPropagation();
      state.selectedType = "text";
      state.selectedId = o.id;
      renderPropertiesPanel();
    });

    contentGroup.appendChild(g);
  }

  btnAddText.onclick = () => {
    pushHistory();
    const id = generateId("text");
    const center = clientToSvg(canvas.clientWidth / 2, canvas.clientHeight / 2);
    state.schema.texts.push({
      id,
      x: center.x,
      y: center.y,
      text: "Comment"
    });
    state.selectedType = "text";
    state.selectedId = id;
    render();
  };

  /* ---------------- Nodes ---------------- */

  function drawNode(node) {
    const { x, y } = node.position;
    const { width: w, height: h } = node.size;

    const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
    g.setAttribute("class", "node");
    g.setAttribute("data-id", node.id);
    g.setAttribute("data-type", "node");

    const selected = state.nodeSelection.has(node.id);
    const fill = selected ? "#b6ffdc" : "#ffffff";

    const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
    rect.setAttribute("x", x - w / 2);
    rect.setAttribute("y", y - h / 2);
    rect.setAttribute("width", w);
    rect.setAttribute("height", h);
    rect.setAttribute("rx", 8);
    rect.setAttribute("fill", fill);
    rect.setAttribute("stroke", "#4a6fa3");
    g.appendChild(rect);

    const label = document.createElementNS("http://www.w3.org/2000/svg", "text");
    label.setAttribute("x", x);
    label.setAttribute("y", y);
    label.setAttribute("text-anchor", "middle");
    label.setAttribute("dominant-baseline", "middle");
    label.setAttribute("font-size", "12");
    label.textContent = node.name || node.id;
    g.appendChild(label);

    g.addEventListener("mousedown", (e) => {
      e.stopPropagation();

      if (state.tool === "arrow") {
        if (!state.nodeSelection.has(node.id)) {
          state.nodeSelection.add(node.id);
        }
      } else {
        state.nodeSelection.clear();
        state.nodeSelection.add(node.id);
      }

      state.selectedType = "node";
      state.selectedId = node.id;
      renderPropertiesPanel();

      // prepare drag of multiple
      state.draggingNodes = true;
      state.dragNodeOffsets = {};
      const p = clientToSvg(e.clientX, e.clientY);
      const idsToMove = state.tool === "arrow"
        ? Array.from(state.nodeSelection)
        : [node.id];

      idsToMove.forEach((id) => {
        const n = getNodeById(id);
        state.dragNodeOffsets[id] = {
          dx: p.x - n.position.x,
          dy: p.y - n.position.y
        };
      });
    });

    contentGroup.appendChild(g);
  }

  /* ---------------- Flows ---------------- */

  function drawFlow(flow) {
    const src = flow.source_node_id ? getNodeById(flow.source_node_id) : null;
    const tgt = flow.target_node_id ? getNodeById(flow.target_node_id) : null;
    if (!src && !tgt) return;

    let startX, startY, endX, endY;

    if (src) {
      startX = src.position.x + src.size.width / 2;
      startY = src.position.y;
    } else {
      startX = tgt.position.x - 200;
      startY = tgt.position.y;
    }

    if (tgt) {
      endX = tgt.position.x - tgt.size.width / 2;
      endY = tgt.position.y;
    } else {
      endX = src.position.x + 200;
      endY = src.position.y;
    }

    const color = flow.color;
    const thickness = Math.max(2, Math.min(10, (flow.value || 1) / 50));
    const arrowLen = 10;

    const dirX = endX - startX;
    const dirY = endY - startY;
    const len = Math.sqrt(dirX * dirX + dirY * dirY) || 1;
    const ux = dirX / len;
    const uy = dirY / len;
    const lineEndX = endX - ux * arrowLen;
    const lineEndY = endY - uy * arrowLen;

    let d;

    if (src && tgt && src.position.x > tgt.position.x + 10) {
      // polyline for right->left
      const loopY = Math.max(src.position.y, tgt.position.y) + 120;
      const p1 = { x: startX + 60, y: startY };
      const p2 = { x: startX + 60, y: loopY };
      const p3 = { x: endX - 60, y: loopY };
      const p4 = { x: endX - 60, y: endY };
      const p5 = { x: endX - arrowLen, y: endY };
      d = `M ${startX} ${startY}
           L ${p1.x} ${p1.y}
           L ${p2.x} ${p2.y}
           L ${p3.x} ${p3.y}
           L ${p4.x} ${p4.y}
           L ${p5.x} ${p5.y}`;
    } else {
      const midX = (startX + lineEndX) / 2;
      d = `M ${startX} ${startY}
           C ${midX} ${startY}, ${midX} ${endY}, ${lineEndX} ${endY}`;
    }

    const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
    path.setAttribute("d", d);
    path.setAttribute("stroke-width", thickness);
    path.setAttribute(
      "stroke",
      state.selectedType === "flow" && state.selectedId === flow.id
        ? darkenColor(color, 0.25)
        : color
    );
    path.setAttribute("fill", "none");
    path.setAttribute("data-id", flow.id);
    path.setAttribute("data-type", "flow");

    path.addEventListener("click", (e) => {
      e.stopPropagation();
      state.selectedType = "flow";
      state.selectedId = flow.id;
      renderPropertiesPanel();
    });

    contentGroup.appendChild(path);

    // arrow
    const baseX = endX - ux * arrowLen;
    const baseY = endY - uy * arrowLen;
    const perpX = -uy;
    const perpY = ux;
    const aw = 6;
    const leftX = baseX + perpX * (aw / 2);
    const leftY = baseY + perpY * (aw / 2);
    const rightX = baseX - perpX * (aw / 2);
    const rightY = baseY - perpY * (aw / 2);

    const arrow = document.createElementNS("http://www.w3.org/2000/svg", "polygon");
    arrow.setAttribute("points", `${endX},${endY} ${leftX},${leftY} ${rightX},${rightY}`);
    arrow.setAttribute("fill", color);
    contentGroup.appendChild(arrow);

    // label
    if (flow.show_label && flow.name) {
      const labelGroup = document.createElementNS("http://www.w3.org/2000/svg", "g");
      const midLX = (startX + lineEndX) / 2;
      const midLY = (startY + lineEndY) / 2;

      const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
      text.setAttribute("x", midLX);
      text.setAttribute("y", midLY - 6);
      text.setAttribute("text-anchor", "middle");
      text.setAttribute("font-size", "11");
      text.setAttribute("fill", "#1f2933");
      text.textContent = flow.name;
      labelGroup.appendChild(text);
      contentGroup.appendChild(labelGroup);

      const bbox = text.getBBox();
      const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
      rect.setAttribute("x", bbox.x - 4);
      rect.setAttribute("y", bbox.y - 2);
      rect.setAttribute("width", bbox.width + 8);
      rect.setAttribute("height", bbox.height + 4);
      rect.setAttribute("fill", "#ffffff");
      rect.setAttribute("stroke", "none");
      labelGroup.insertBefore(rect, text);
    }
  }

  /* ---------------- Properties panel ---------------- */

function renderPropertiesPanel() {
  const collapsed = propertiesPanel.classList.contains("collapsed");
  propertiesPanel.innerHTML = "";

  // Кнопка сворачивания / разворачивания
  const collapseBtn = document.createElement("button");
  collapseBtn.className = "collapse-toggle";
  collapseBtn.textContent = collapsed ? "«" : "»";
  collapseBtn.title = collapsed ? "Show properties" : "Hide properties";
  collapseBtn.onclick = () => {
    propertiesPanel.classList.toggle("collapsed");
    renderPropertiesPanel();
  };
  propertiesPanel.appendChild(collapseBtn);

  // Если панель свернута — больше ничего не рисуем
  if (collapsed) return;

  if (!state.selectedType || state.selectedType === "schema") {
    renderSchemaProps();
  } else if (state.selectedType === "node") {
    const node = getNodeById(state.selectedId);
    if (node) renderNodeProps(node);
  } else if (state.selectedType === "flow") {
    const flow = getFlowById(state.selectedId);
    if (flow) renderFlowProps(flow);
  } else if (state.selectedType === "text") {
    const txt = getTextById(state.selectedId);
    if (txt) renderTextProps(txt);
  }
}

  function renderSchemaProps() {
    const s = state.schema.schema;
    const section = document.createElement("div");
    section.className = "section";

    const h = document.createElement("h3");
    h.textContent = "Schema";
    section.appendChild(h);

    section.appendChild(fieldText("Title", s.title, (v) => {
      s.title = v;
      updateTimestamp();
      refreshTitle();
    }));

    section.appendChild(fieldTextarea("Description", s.description, (v) => {
      s.description = v;
      updateTimestamp();
    }));

    section.appendChild(fieldText("Version", s.version, (v) => {
      s.version = v;
      updateTimestamp();
    }));

    const info = document.createElement("div");
    info.style.fontSize = "11px";
    info.textContent = "ID: " + s.id;
    section.appendChild(info);

    propertiesPanel.appendChild(section);
  }

  function renderNodeProps(node) {
    const section = document.createElement("div");
    section.className = "section";

    const header = document.createElement("div");
    header.className = "section-header";
    const h = document.createElement("h3");
    h.textContent = "Converter";
    header.appendChild(h);

    const del = document.createElement("button");
    del.className = "btn-small";
    del.textContent = "Delete";
    del.onclick = () => {
      pushHistory();
      deleteNode(node.id, false);
    };
    header.appendChild(del);

    section.appendChild(header);

    const info = document.createElement("div");
    info.style.fontSize = "11px";
    info.textContent = "ID: " + node.id;
    section.appendChild(info);

    section.appendChild(fieldText("Name", node.name, (v) => {
      node.name = v;
      updateTimestamp();
      renderDiagram();
    }));

    section.appendChild(fieldTextarea("Description", node.description, (v) => {
      node.description = v;
      updateTimestamp();
    }));

    section.appendChild(fieldText("Link", node.link || "", (v) => {
      node.link = v;
      updateTimestamp();
    }));

    // Flow buttons
    const btnRow = document.createElement("div");
    btnRow.style.display = "flex";
    btnRow.style.flexWrap = "wrap";
    btnRow.style.gap = "4px";
    btnRow.style.marginTop = "6px";

    const btnNext = document.createElement("button");
    btnNext.className = "btn-small";
    btnNext.textContent = "Add next converter";
    btnNext.onclick = () => {
      pushHistory();
      addNextConverter(node.id);
    };
    btnRow.appendChild(btnNext);

    const btnExtOut = document.createElement("button");
    btnExtOut.className = "btn-small";
    btnExtOut.textContent = "Add outgoing external";
    btnExtOut.onclick = () => {
      pushHistory();
      addExternalOutFlow(node.id);
    };
    btnRow.appendChild(btnExtOut);

    const btnExtIn = document.createElement("button");
    btnExtIn.className = "btn-small";
    btnExtIn.textContent = "Add incoming external";
    btnExtIn.onclick = () => {
      pushHistory();
      addExternalInFlow(node.id);
    };
    btnRow.appendChild(btnExtIn);

    section.appendChild(btnRow);

    // Element types
    const elemsSection = document.createElement("div");
    elemsSection.className = "section";
    const etTitle = document.createElement("h3");
    etTitle.textContent = "Element types";
    elemsSection.appendChild(etTitle);

    const list = document.createElement("div");
    const elems = getElementsInNode(node.id);
    if (!elems.length) {
      list.textContent = "No element types.";
    } else {
      elems.forEach(e => {
        const row = document.createElement("div");
        row.style.display = "flex";
        row.style.alignItems = "center";
        row.style.gap = "4px";
        const span = document.createElement("span");
        const signLabel = e.kind === "anti_value" ? "Anti-value" : "Value";
        span.textContent = `${e.name} [${signLabel}] ${e.quantity}${e.unit || ""}`;
        row.appendChild(span);

        const edit = document.createElement("button");
        edit.className = "btn-small";
        edit.textContent = "Edit";
        edit.onclick = () => showElementTypeEditor(node, e);
        row.appendChild(edit);

        const delE = document.createElement("button");
        delE.className = "btn-small";
        delE.textContent = "✕";
        delE.onclick = () => {
          pushHistory();
          state.schema.elements = state.schema.elements.filter(el => el.id !== e.id);
          renderPropertiesPanel();
        };
        row.appendChild(delE);

        list.appendChild(row);
      });
    }
    elemsSection.appendChild(list);

    // Add new type
    const addBtn = document.createElement("button");
    addBtn.className = "btn-small";
    addBtn.textContent = "Add element type";
    addBtn.style.marginTop = "4px";
    addBtn.onclick = () => showElementTypeEditor(node, null);
    elemsSection.appendChild(addBtn);

    propertiesPanel.appendChild(section);
    propertiesPanel.appendChild(elemsSection);
  }

  function showElementTypeEditor(node, elem) {
    const dlg = document.createElement("div");
    dlg.className = "section";

    const h = document.createElement("h3");
    h.textContent = elem ? "Edit element type" : "Add element type";
    dlg.appendChild(h);

    const nameField = fieldText("Name", elem ? elem.name : "", () => {});
    const kindField = fieldSelect("Type", elem ? elem.kind : "value", [
      ["value", "Value"],
      ["anti_value", "Anti-value"]
    ], () => {});
    const qtyField = fieldNumber("Qty", elem ? elem.quantity : 0, () => {});
    const unitField = fieldText("Unit", elem ? (elem.unit || "Æ") : "Æ", () => {});

    dlg.appendChild(nameField);
    dlg.appendChild(kindField);
    dlg.appendChild(qtyField);
    dlg.appendChild(unitField);

    const save = document.createElement("button");
    save.className = "btn-small";
    save.textContent = "Save";
    save.onclick = () => {
      const name = nameField.querySelector("input").value.trim();
      const kind = kindField.querySelector("select").value;
      const qty = parseFloat(qtyField.querySelector("input").value) || 0;
      const unit = unitField.querySelector("input").value || "Æ";
      if (!name) return;

      pushHistory();

      if (elem) {
        elem.name = name;
        elem.kind = kind;
        elem.sign = kind === "anti_value" ? -1 : 1;
        elem.quantity = qty;
        elem.unit = unit;
      } else {
        state.schema.elements.push({
          id: generateId("elem"),
          name,
          kind,
          sign: kind === "anti_value" ? -1 : 1,
          quantity: qty,
          unit,
          location: "in_node",
          node_id: node.id,
          flow_id: null,
          attributes: {}
        });
      }

      updateTimestamp();
      renderPropertiesPanel();
    };

    dlg.appendChild(save);

    propertiesPanel.appendChild(dlg);
  }

function renderFlowProps(flow) {
  const section = document.createElement("div");
  section.className = "section";

  const header = document.createElement("div");
  header.className = "section-header";
  const h = document.createElement("h3");
  h.textContent = "Flow";
  header.appendChild(h);

  const del = document.createElement("button");
  del.className = "btn-small";
  del.textContent = "Delete";
  del.onclick = () => {
    pushHistory();
    deleteFlow(flow.id, false);
  };
  header.appendChild(del);

  section.appendChild(header);

  const info = document.createElement("div");
  info.style.fontSize = "11px";
  info.textContent = "ID: " + flow.id;
  section.appendChild(info);

  // Name
  section.appendChild(fieldText("Name", flow.name, (v) => {
    flow.name = v;
    updateTimestamp();
    renderDiagram();
  }));

  // Type
  section.appendChild(fieldSelect("Type", flow.type, [
    ["external_in", "External in"],
    ["external_out", "External out"],
    ["internal", "Internal"]
  ], (v) => {
    flow.type = v;
    updateTimestamp();
    renderDiagram();
  }));

  // Value / Unit
  section.appendChild(fieldNumber("Value", flow.value || 0, (v) => {
    flow.value = v;
    updateTimestamp();
    renderDiagram();
  }));

  section.appendChild(fieldText("Unit", flow.unit || "Æ", (v) => {
    flow.unit = v;
    updateTimestamp();
  }));

  // Color
  section.appendChild(fieldColor("Color", flow.color, (v) => {
    flow.color = v;
    updateTimestamp();
    renderDiagram();
  }));

  // Show label
  section.appendChild(fieldCheckbox("Show name on diagram", flow.show_label, (v) => {
    flow.show_label = v;
    updateTimestamp();
    renderDiagram();
  }));

  // --- Source / Target node selectors ---

  const nodeOptions = [["", "(External)"]].concat(
    state.schema.nodes.map(n => [n.id, n.name || n.id])
  );

  section.appendChild(fieldSelect("Source node", flow.source_node_id || "", nodeOptions, (val) => {
    flow.source_node_id = val || null;
    autoTypeFlow(flow);
    updateTimestamp();
    renderDiagram();
    renderPropertiesPanel();
  }));

  section.appendChild(fieldSelect("Target node", flow.target_node_id || "", nodeOptions, (val) => {
    flow.target_node_id = val || null;
    autoTypeFlow(flow);
    updateTimestamp();
    renderDiagram();
    renderPropertiesPanel();
  }));

  // Direction info
  const dir = document.createElement("div");
  dir.style.fontSize = "11px";
  dir.style.marginTop = "4px";
  const srcName = flow.source_node_id
    ? (getNodeById(flow.source_node_id)?.name || flow.source_node_id)
    : "External";
  const tgtName = flow.target_node_id
    ? (getNodeById(flow.target_node_id)?.name || flow.target_node_id)
    : "External";
  dir.textContent = `Direction: ${srcName} → ${tgtName}`;
  section.appendChild(dir);

  propertiesPanel.appendChild(section);

  // --- Element types viewer for this flow ---

  const etSection = document.createElement("div");
  etSection.className = "section";
  const etHeader = document.createElement("h3");
  etHeader.textContent = "Element types in this flow";
  etSection.appendChild(etHeader);

  let converterForElements = null;
  if (flow.source_node_id) {
    // Internal / external_out – берём типы у source
    converterForElements = getNodeById(flow.source_node_id);
  } else if (!flow.source_node_id && flow.target_node_id) {
    // External_in – типы у target
    converterForElements = getNodeById(flow.target_node_id);
  }

  const list = document.createElement("div");
  if (!converterForElements) {
    list.textContent = "No related converter for element types.";
  } else {
    const elems = getElementsInNode(converterForElements.id);
    if (!elems.length) {
      list.textContent = "Converter has no element types.";
    } else {
      elems.forEach(e => {
        const row = document.createElement("div");
        row.style.fontSize = "12px";
        const signLabel = e.kind === "anti_value" ? "Anti-value" : "Value";
        row.textContent = `${e.name} [${signLabel}] ${e.quantity}${e.unit || ""}`;
        list.appendChild(row);
      });
    }
  }
  etSection.appendChild(list);

  const hint = document.createElement("div");
  hint.style.fontSize = "11px";
  hint.style.color = "#718096";
  hint.style.marginTop = "4px";
  hint.textContent = "Element types are defined in the converter and cannot be edited from the flow.";
  etSection.appendChild(hint);

  propertiesPanel.appendChild(etSection);
}

  function renderTextProps(txt) {
    const section = document.createElement("div");
    section.className = "section";

    const h = document.createElement("h3");
    h.textContent = "Text";
    section.appendChild(h);

    const info = document.createElement("div");
    info.style.fontSize = "11px";
    info.textContent = "ID: " + txt.id;
    section.appendChild(info);

    section.appendChild(fieldText("Text", txt.text || "Comment", (v) => {
      txt.text = v;
      updateTimestamp();
      renderDiagram();
    }));

    propertiesPanel.appendChild(section);
  }

  /* ---------------- Field helpers ---------------- */

  function fieldText(label, value, onChange) {
    const f = document.createElement("div");
    f.className = "field";
    const l = document.createElement("label");
    l.textContent = label;
    f.appendChild(l);
    const i = document.createElement("input");
    i.type = "text";
    i.value = value || "";
    i.oninput = () => onChange(i.value);
    f.appendChild(i);
    return f;
  }

  function fieldTextarea(label, value, onChange) {
    const f = document.createElement("div");
    f.className = "field";
    const l = document.createElement("label");
    l.textContent = label;
    f.appendChild(l);
    const t = document.createElement("textarea");
    t.value = value || "";
    t.oninput = () => onChange(t.value);
    f.appendChild(t);
    return f;
  }

  function fieldNumber(label, value, onChange) {
    const f = document.createElement("div");
    f.className = "field";
    const l = document.createElement("label");
    l.textContent = label;
    f.appendChild(l);
    const i = document.createElement("input");
    i.type = "number";
    i.value = value;
    i.oninput = () => onChange(parseFloat(i.value) || 0);
    f.appendChild(i);
    return f;
  }

  function fieldSelect(label, value, options, onChange) {
    const f = document.createElement("div");
    f.className = "field";
    const l = document.createElement("label");
    l.textContent = label;
    f.appendChild(l);
    const s = document.createElement("select");
    options.forEach(([val, text]) => {
      const opt = document.createElement("option");
      opt.value = val;
      opt.textContent = text;
      if (val === value) opt.selected = true;
      s.appendChild(opt);
    });
    s.onchange = () => onChange(s.value);
    f.appendChild(s);
    return f;
  }

  function fieldColor(label, value, onChange) {
    const f = document.createElement("div");
    f.className = "field";
    const l = document.createElement("label");
    l.textContent = label;
    f.appendChild(l);
    const i = document.createElement("input");
    i.type = "color";
    i.value = value;
    i.oninput = () => onChange(i.value);
    f.appendChild(i);
    return f;
  }

  function fieldCheckbox(label, checked, onChange) {
    const f = document.createElement("div");
    f.className = "field";
    const l = document.createElement("label");
    l.textContent = label;
    f.appendChild(l);
    const i = document.createElement("input");
    i.type = "checkbox";
    i.checked = checked;
    i.onchange = () => onChange(i.checked);
    f.appendChild(i);
    return f;
  }

  /* ---------------- Schema operations ---------------- */

  function deleteNode(nodeId, push = true) {
    if (push) pushHistory();
    state.schema.flows = state.schema.flows.filter(
      f => f.source_node_id !== nodeId && f.target_node_id !== nodeId
    );
    state.schema.elements = state.schema.elements.filter(
      e => !(e.location === "in_node" && e.node_id === nodeId)
    );
    state.schema.nodes = state.schema.nodes.filter(n => n.id !== nodeId);
    state.nodeSelection.delete(nodeId);
    selectSchema();
    render();
  }

  function deleteFlow(flowId, push = true) {
    if (push) pushHistory();
    state.schema.flows = state.schema.flows.filter(f => f.id !== flowId);
    selectSchema();
    render();
  }

  function addNextConverter(nodeId) {
    const node = getNodeById(nodeId);
    if (!node) return;
    const id = generateId("node");
    const newNode = {
      id,
      name: "Converter " + id,
      position: { x: node.position.x + 250, y: node.position.y },
      size: { width: 120, height: 60 },
      description: "",
      link: "",
      attributes: {}
    };
    state.schema.nodes.push(newNode);
    state.schema.flows.push({
      id: generateId("flow"),
      name: "Flow " + id,
      type: "internal",
      source_node_id: node.id,
      target_node_id: newNode.id,
      value: 50,
      unit: "Æ",
      color: getNextFlowColor(),
      show_label: false,
      attributes: {}
    });
    updateTimestamp();
    render();
  }

  function addExternalOutFlow(nodeId) {
    const node = getNodeById(nodeId);
    if (!node) return;
    state.schema.flows.push({
      id: generateId("flow"),
      name: "External Out",
      type: "external_out",
      source_node_id: node.id,
      target_node_id: null,
      value: 50,
      unit: "Æ",
      color: getNextFlowColor(),
      show_label: false,
      attributes: {}
    });
    updateTimestamp();
    render();
  }

  function addExternalInFlow(nodeId) {
    const node = getNodeById(nodeId);
    if (!node) return;
    state.schema.flows.push({
      id: generateId("flow"),
      name: "External In",
      type: "external_in",
      source_node_id: null,
      target_node_id: node.id,
      value: 50,
      unit: "Æ",
      color: getNextFlowColor(),
      show_label: false,
      attributes: {}
    });
    updateTimestamp();
    render();
  }

  /* ---------------- Help popup ---------------- */

  btnHelp.onclick = () => helpModal.classList.remove("hidden");
  helpClose.onclick = () => helpModal.classList.add("hidden");
  helpModal.addEventListener("click", (e) => {
    if (e.target === helpModal) helpModal.classList.add("hidden");
  });

  /* ---------------- Schema menu & export PNG ---------------- */

  schemaMenuButton.onclick = (e) => {
    e.stopPropagation();
    schemaMenu.classList.toggle("open");
  };
  document.addEventListener("click", (e) => {
    if (!schemaMenu.contains(e.target)) schemaMenu.classList.remove("open");
  });

  btnNew.onclick = () => {
    schemaMenu.classList.remove("open");
    if (!confirm("Discard current schema and create a new one?")) return;
    pushHistory();
    state.schema = normalizeSchema(createEmptySchema());
    selectSchema();
    centerView();
    render();
  };

  btnOpen.onclick = () => {
    schemaMenu.classList.remove("open");
    state.fileMode = "open";
    fileInput.value = "";
    fileInput.click();
  };

  btnImport.onclick = () => {
    schemaMenu.classList.remove("open");
    state.fileMode = "import";
    fileInput.value = "";
    fileInput.click();
  };

  btnSaveFile.onclick = () => {
    schemaMenu.classList.remove("open");
    const data = JSON.stringify(state.schema, null, 2);
    const blob = new Blob([data], { type: "application/json" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = (state.schema.schema.title || "schema") + ".json";
    a.click();
    URL.revokeObjectURL(url);
  };

  btnExportPng.onclick = () => {
    schemaMenu.classList.remove("open");
    exportPng();
  };

fileInput.onchange = (e) => {
  const file = e.target.files[0];
  if (!file) return;
  const reader = new FileReader();
  reader.onload = (ev) => {
    try {
      const obj = JSON.parse(ev.target.result);
      if (state.fileMode === "open") {
        // Полная замена схемы
        pushHistory();
        state.schema = normalizeSchema(obj);
        selectSchema();
        centerView();
        render();
      } else if (state.fileMode === "import") {
        // Импорт "внутрь" текущей схемы с префиксами
        pushHistory();
        importWithPrefix(normalizeSchema(obj));
      }
    } catch {
      alert("Invalid JSON");
    }
  };
  reader.readAsText(file);
};

function importWithPrefix(imported) {
  // Наборы уже существующих ID
  const existingNodeIds = new Set(state.schema.nodes.map(n => n.id));
  const existingFlowIds = new Set(state.schema.flows.map(f => f.id));
  const existingElemIds = new Set(state.schema.elements.map(e => e.id));
  const existingTextIds = new Set(state.schema.texts.map(t => t.id));

  // Префикс для импортируемой схемы
  const prefix = "imp" + Math.random().toString(36).slice(2, 6) + "_";

  const nodeIdMap = {};
  const flowIdMap = {};

  // --- Ноды ---
  (imported.nodes || []).forEach(n => {
    let newId = n.id;
    if (existingNodeIds.has(newId)) {
      newId = prefix + newId;
    }
    existingNodeIds.add(newId);
    nodeIdMap[n.id] = newId;

    const clone = Object.assign({}, n, { id: newId });
    state.schema.nodes.push(clone);
  });

  // --- Потоки ---
  (imported.flows || []).forEach(f => {
    let newId = f.id;
    if (existingFlowIds.has(newId)) {
      newId = prefix + newId;
    }
    existingFlowIds.add(newId);
    flowIdMap[f.id] = newId;

    const clone = Object.assign({}, f, {
      id: newId,
      source_node_id: f.source_node_id
        ? (nodeIdMap[f.source_node_id] || f.source_node_id)
        : null,
      target_node_id: f.target_node_id
        ? (nodeIdMap[f.target_node_id] || f.target_node_id)
        : null
    });

    if (!clone.color) clone.color = getNextFlowColor();
    if (typeof clone.show_label !== "boolean") clone.show_label = false;
    autoTypeFlow(clone);

    state.schema.flows.push(clone);
  });

  // --- Элементы (типы ценности) ---
  (imported.elements || []).forEach(e => {
    let newId = e.id;
    if (existingElemIds.has(newId)) {
      newId = prefix + newId;
    }
    existingElemIds.add(newId);

    const clone = Object.assign({}, e, {
      id: newId,
      node_id: e.node_id ? (nodeIdMap[e.node_id] || e.node_id) : null,
      flow_id: e.flow_id ? (flowIdMap[e.flow_id] || e.flow_id) : null
    });

    state.schema.elements.push(clone);
  });

  // --- Текстовые объекты ---
  (imported.texts || []).forEach(t => {
    let newId = t.id;
    if (existingTextIds.has(newId)) {
      newId = prefix + newId;
    }
    existingTextIds.add(newId);

    const clone = Object.assign(
      {
        x: 0,
        y: 0,
        text: "Comment"
      },
      t,
      { id: newId }
    );

    state.schema.texts.push(clone);
  });

  updateTimestamp();
  render();
}

  function exportPng() {
    if (!contentGroup) {
      alert("Nothing to export");
      return;
    }
    const bbox = contentGroup.getBBox();
    if (!bbox.width || !bbox.height) {
      alert("Nothing to export");
      return;
    }

    const margin = 50;
    const width = bbox.width + margin * 2;
    const height = bbox.height + margin * 2;
    const viewBoxX = bbox.x - margin;
    const viewBoxY = bbox.y - margin;

    const svgNS = "http://www.w3.org/2000/svg";
    const exportSvg = document.createElementNS(svgNS, "svg");
    exportSvg.setAttribute("xmlns", svgNS);
    exportSvg.setAttribute("width", width);
    exportSvg.setAttribute("height", height);
    exportSvg.setAttribute("viewBox", `${viewBoxX} ${viewBoxY} ${width} ${height}`);

    const clonedContent = contentGroup.cloneNode(true);
    exportSvg.appendChild(clonedContent);

    const serializer = new XMLSerializer();
    const svgData = serializer.serializeToString(exportSvg);
    const blob = new Blob([svgData], { type: "image/svg+xml;charset=utf-8" });
    const url = URL.createObjectURL(blob);

    const img = new Image();
    img.onload = () => {
      const c = document.createElement("canvas");
      c.width = width;
      c.height = height;
      const ctx = c.getContext("2d");
      ctx.fillStyle = "#ffffff";
      ctx.fillRect(0, 0, width, height);
      ctx.drawImage(img, 0, 0);
      URL.revokeObjectURL(url);
      const pngUrl = c.toDataURL("image/png");
      const a = document.createElement("a");
      a.href = pngUrl;
      a.download = (state.schema.schema.title || "schema") + ".png";
      a.click();
    };
    img.src = url;
  }

  /* ---------------- Selection / mouse ---------------- */

  function selectSchema() {
    state.selectedType = "schema";
    state.selectedId = state.schema.schema.id;
    state.nodeSelection.clear();
  }

  toolHandBtn.onclick = () => {
    state.tool = "hand";
    toolHandBtn.classList.add("active");
    toolArrowBtn.classList.remove("active");
  };
  toolArrowBtn.onclick = () => {
    state.tool = "arrow";
    toolArrowBtn.classList.add("active");
    toolHandBtn.classList.remove("active");
  };

  canvas.addEventListener("mousedown", (e) => {
    if (e.button !== 0) return;

    if (state.tool === "arrow") {
      state.selecting = true;
      state.selectionAdditive = e.shiftKey || e.ctrlKey || e.metaKey;
      const p = clientToSvg(e.clientX, e.clientY);
      state.selectionStart = p;
      if (state.selectionRect) state.selectionRect.remove();
      const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
      rect.setAttribute("class", "selection-rect");
      rect.setAttribute("x", p.x);
      rect.setAttribute("y", p.y);
      rect.setAttribute("width", 0);
      rect.setAttribute("height", 0);
      viewport.appendChild(rect);
      state.selectionRect = rect;
      return;
    }

    if (state.tool === "hand") {
      state.draggingPan = true;
      state.panMoved = false;
      state.dragStart = { x: e.clientX, y: e.clientY };
      state.panStart = { ...state.pan };
      canvas.classList.add("dragging");
    }
  });

  canvas.addEventListener("mousemove", (e) => {
    if (state.selecting && state.selectionRect) {
      const p = clientToSvg(e.clientX, e.clientY);
      const x = Math.min(p.x, state.selectionStart.x);
      const y = Math.min(p.y, state.selectionStart.y);
      const w = Math.abs(p.x - state.selectionStart.x);
      const h = Math.abs(p.y - state.selectionStart.y);
      state.selectionRect.setAttribute("x", x);
      state.selectionRect.setAttribute("y", y);
      state.selectionRect.setAttribute("width", w);
      state.selectionRect.setAttribute("height", h);
      return;
    }

    if (state.draggingNodes && state.dragNodeOffsets) {
      const p = clientToSvg(e.clientX, e.clientY);
      Object.keys(state.dragNodeOffsets).forEach((id) => {
        const n = getNodeById(id);
        if (!n) return;
        const o = state.dragNodeOffsets[id];
        n.position.x = snap(p.x - o.dx);
        n.position.y = snap(p.y - o.dy);
      });
      updateTimestamp();
      renderDiagram();
      return;
    }

    if (state.draggingPan) {
      const dx = e.clientX - state.dragStart.x;
      const dy = e.clientY - state.dragStart.y;
      if (Math.abs(dx) > 2 || Math.abs(dy) > 2) state.panMoved = true;
      state.pan.x = state.panStart.x + dx;
      state.pan.y = state.panStart.y + dy;
      applyTransform();
    }
  });

  window.addEventListener("mouseup", () => {
    if (state.selecting && state.selectionRect) {
      const rx = parseFloat(state.selectionRect.getAttribute("x"));
      const ry = parseFloat(state.selectionRect.getAttribute("y"));
      const rw = parseFloat(state.selectionRect.getAttribute("width"));
      const rh = parseFloat(state.selectionRect.getAttribute("height"));
      state.selectionRect.remove();
      state.selectionRect = null;

      const selected = [];
      state.schema.nodes.forEach(node => {
        const { x, y } = node.position;
        const { width, height } = node.size;
        const nx = x - width / 2;
        const ny = y - height / 2;
        const intersect =
          nx < rx + rw &&
          nx + width > rx &&
          ny < ry + rh &&
          ny + height > ry;
        if (intersect) selected.push(node.id);
      });

      if (!state.selectionAdditive) state.nodeSelection.clear();
      selected.forEach(id => state.nodeSelection.add(id));
      state.selectionJustFinished = true;
      render();
      state.selecting = false;
    }

    if (state.draggingNodes && state.dragNodeOffsets) {
      pushHistory();
    }
    state.draggingNodes = false;
    state.dragNodeOffsets = null;

    state.draggingPan = false;
    canvas.classList.remove("dragging");
  });

  canvas.addEventListener("click", (e) => {
    if (state.selectionJustFinished) {
      state.selectionJustFinished = false;
      return;
    }
    if (state.panMoved) {
      state.panMoved = false;
      return;
    }
    const t = e.target;
    if (t === canvas || t === viewport || t.id === "grid" || t.id === "content") {
      selectSchema();
      renderPropertiesPanel();
      renderDiagram();
    }
  });

  /* ---------------- PNG export ---------------- */

  function exportPng() {
    if (!contentGroup) {
      alert("Nothing to export");
      return;
    }
    const bbox = contentGroup.getBBox();
    if (!bbox.width || !bbox.height) {
      alert("Nothing to export");
      return;
    }

    const margin = 50;
    const width = bbox.width + margin * 2;
    const height = bbox.height + margin * 2;
    const viewBoxX = bbox.x - margin;
    const viewBoxY = bbox.y - margin;

    const svgNS = "http://www.w3.org/2000/svg";
    const exportSvg = document.createElementNS(svgNS, "svg");
    exportSvg.setAttribute("xmlns", svgNS);
    exportSvg.setAttribute("width", width);
    exportSvg.setAttribute("height", height);
    exportSvg.setAttribute("viewBox", `${viewBoxX} ${viewBoxY} ${width} ${height}`);

    const clonedContent = contentGroup.cloneNode(true);
    exportSvg.appendChild(clonedContent);

    const serializer = new XMLSerializer();
    const svgData = serializer.serializeToString(exportSvg);
    const blob = new Blob([svgData], { type: "image/svg+xml;charset=utf-8" });
    const url = URL.createObjectURL(blob);

    const img = new Image();
    img.onload = () => {
      const c = document.createElement("canvas");
      c.width = width;
      c.height = height;
      const ctx = c.getContext("2d");
      ctx.fillStyle = "#ffffff";
      ctx.fillRect(0, 0, width, height);
      ctx.drawImage(img, 0, 0);
      URL.revokeObjectURL(url);
      const pngUrl = c.toDataURL("image/png");
      const a = document.createElement("a");
      a.href = pngUrl;
      a.download = (state.schema.schema.title || "schema") + ".png";
      a.click();
    };
    img.src = url;
  }

  /* ---------------- Init ---------------- */

  function init() {
    state.schema = normalizeSchema(createEmptySchema());
    state.tool = "hand";
    toolHandBtn.classList.add("active");
    centerView();
    selectSchema();
    render();
  }

  init();
})();